<?php

namespace Ebizmarts\SagePaySuite\Test\Integration;

use Ebizmarts\SagePaySuite\Model\Config;
use Ebizmarts\SagePaySuite\Helper\Fraud;
use Ebizmarts\SagePaySuite\Api\SagePayData\FraudScreenResponseInterfaceFactory;
use Ebizmarts\SagePaySuite\Api\SagePayData\FraudScreenResponseInterface;
use Magento\Sales\Model\Order\Payment\Transaction;
use Magento\Sales\Model\Order\Payment\Transaction\Repository;
use Magento\Sales\Api\OrderRepositoryInterface;
use Magento\Sales\Model\Order;
use Magento\Sales\Model\ResourceModel\Order\CollectionFactory;
use Magento\TestFramework\Helper\Bootstrap;
use Magento\TestFramework\ObjectManager;

class FraudCheckConfigTest extends \PHPUnit\Framework\TestCase
{
    /** @var ObjectManager $objectManager */
    private $objectManager;

    /** @var CollectionFactory */
    private $orderCollectionFactory;

    /** @var Config */
    private $config;

    /** @var Fraud */
    private $fraud;

    /** @var Repository */
    private $transactionRepository;

    /** @var FraudScreenResponseInterfaceFactory */
    private $fraudScreenResponseInterfaceFactory;

    /** @var OrderRepositoryInterface */
    private $orderRepositoryInterface;

    /**
     * @throws \Exception
     */
    public function setUp() : void
    {
        $this->objectManager = Bootstrap::getObjectManager();
        $this->orderCollectionFactory = $this->objectManager->create(
            CollectionFactory::class
        );
        $this->config = $this->objectManager->create(
            Config::class
        );
        $this->fraud = $this->objectManager->create(
            Fraud::class
        );
        $this->transactionRepository = $this->objectManager->create(
            Repository::class
        );
        $this->orderRepositoryInterface = $this->objectManager->create(
            OrderRepositoryInterface::class
        );
        $this->fraudScreenResponseInterfaceFactory = $this->objectManager->create(
            FraudScreenResponseInterfaceFactory::class
        );
    }

    /**
     * @throws \Magento\Framework\Exception\LocalizedException
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function testFraudCheckOK()
    {
        $customerEmailId = "testFraudCheck@ebizmarts.com";
        $customerOrder = $this->orderCollectionFactory->create()
            ->addAttributeToFilter('customer_email', $customerEmailId)
            ->load();
        $orders = $customerOrder->getData();

        foreach ($orders as $order) {
            $orderId = $order['entity_id'];
            /** @var Order $orderModel */
            $orderModel = $this->orderRepositoryInterface->get($orderId);
            $payment = $orderModel->getPayment();
            $response = $this->getResponse(42, 'OK');
            $transaction = $this->transactionRepository
                ->getByTransactionId($payment->getLastTransId(), $payment->getId(), $orderId);

            if ($transaction->getTxnType() == Transaction::TYPE_AUTH) {
                $autoInvoiceActioned = $this->fraud->processAutoInvoiceForTests(
                    $transaction,
                    $payment,
                    $response
                );

                $this->assertNotEmpty($autoInvoiceActioned);
            }
        }
    }

    /**
     * @return void
     * @throws \Magento\Framework\Exception\InputException
     */
    public function testFraudCheckREJECT()
    {
        $customerEmailId = "testFraudCheck@ebizmarts.com";
        $customerOrder = $this->orderCollectionFactory->create()
            ->addAttributeToFilter('customer_email', $customerEmailId)
            ->load();
        $orders = $customerOrder->getData();

        foreach ($orders as $order) {
            $orderId = $order['entity_id'];
            /** @var Order $orderModel */
            $orderModel = $this->orderRepositoryInterface->get($orderId);
            $payment = $orderModel->getPayment();
            $response = $this->getResponse(101, 'REJECT');
            $transaction = $this->transactionRepository
                ->getByTransactionId($payment['last_trans_id'], $payment['entity_id'], $orderId);

            $autoInvoiceActioned = $this->fraud->processAutoInvoiceForTests(
                $transaction,
                $payment,
                $response
            );

            $this->assertFalse($autoInvoiceActioned);
        }
    }

    /**
     * @return FraudScreenResponseInterface|mixed
     */
    private function getResponse($score, $action)
    {
        /** @var FraudScreenResponseInterface $response */
        $response = $this->fraudScreenResponseInterfaceFactory->create();
        $response->setErrorCode(0000);
        $response->setFraudProviderName('FSG');
        $response->setThirdmanId(6668815754);
        $response->setThirdmanScore($score);
        $response->setThirdmanAction($action);

        return $response;
    }
}
