<?php

namespace Ebizmarts\SagePaySuite\Test;

use Ebizmarts\SagePaySuite\Api\PiManagementInterfaceFactory;
use Ebizmarts\SagePaySuite\Api\PiManagementInterface;
use Ebizmarts\SagePaySuite\Api\Data\PiRequest;
use Ebizmarts\SagePaySuite\Api\Data\PiRequestFactory;
use Ebizmarts\SagePaySuite\Api\PiMerchantInterfaceFactory;
use Ebizmarts\SagePaySuite\Api\PiMerchantInterface;
use Ebizmarts\SagePaySuite\Api\Data\ResultInterface;
use Ebizmarts\SagePaySuite\Api\Data\PiResultInterface;
use Magento\Catalog\Model\Product;
use Magento\Customer\Api\CustomerRepositoryInterface;
use Magento\Customer\Model\CustomerFactory;
use Magento\Framework\App\Cache\Manager as CacheManager;
use Magento\Framework\App\Helper\AbstractHelper;
use Magento\Framework\App\Helper\Context;
use Magento\Framework\Data\Form\FormKey;
use Magento\Quote\Model\Quote;
use Magento\Quote\Model\QuoteFactory;
use Magento\Quote\Model\QuoteManagement;
use Magento\Sales\Model\Service\OrderService;
use Magento\Store\Model\Store;
use Magento\Store\Model\StoreManagerInterface;
use Magento\Framework\HTTP\Adapter\Curl;
use Magento\Config\Model\Config;
use Laminas\Http\Request as HttpRequest;
use Laminas\Http\Response as HttpResponse;

class PlaceMagentoOrder extends AbstractHelper
{
    private const TEST_CC_NUMBER = "4929000005559";
    private const TEST_CC_CV2 = "123";
    private const TEST_CC_TYPE = "VI";
    private const TEST_CC_EXPIRY = "0329";

    /** @var StoreManagerInterface $storeManager */
    private $storeManager;

    /** @var Product $product */
    private $product;

    /** @var FormKey $formkey */
    private $formkey;

    /** @var QuoteFactory $quote */
    private $quote;

    /** @var QuoteManagement $quoteManagement */
    private $quoteManagement;

    /** @var CustomerFactory $customerFactory */
    private $customerFactory;

    /** @var CustomerRepositoryInterface $customerRepository */
    private $customerRepository;

    /** @var OrderService $orderService */
    private $orderService;

    /** @var PiManagementInterfaceFactory $piManagementInterfaceFactory */
    private $piManagementInterfaceFactory;

    /** @var PiRequestFactory $piRequestFactory */
    private $piRequestFactory;

    /** @var PiMerchantInterfaceFactory $piMerchantInterfaceFactory */
    private $piMerchantInterfaceFactory;

    /** @var Config $config */
    private $config;

    /** @var Curl $curl */
    private $curl;

    /** @var CacheManager */
    private $cacheManager;

    /**
     * PlaceMagentoOrder constructor.
     * @param Context $context
     * @param StoreManagerInterface $storeManager
     * @param Product $product
     * @param FormKey $formkey
     * @param QuoteFactory $quote
     * @param QuoteManagement $quoteManagement
     * @param CustomerFactory $customerFactory
     * @param CustomerRepositoryInterface $customerRepository
     * @param OrderService $orderService
     * @param PiManagementInterfaceFactory $piManagementInterfaceFactory
     * @param PiRequestFactory $piRequestFactory
     * @param PiMerchantInterfaceFactory $piMerchantInterfaceFactory
     * @param Curl $curl
     * @param Config $config
     * @param CacheManager $cacheManager
     */
    public function __construct(
        Context $context,
        StoreManagerInterface $storeManager,
        Product $product,
        FormKey $formkey,
        QuoteFactory $quote,
        QuoteManagement $quoteManagement,
        CustomerFactory $customerFactory,
        CustomerRepositoryInterface $customerRepository,
        OrderService $orderService,
        PiManagementInterfaceFactory $piManagementInterfaceFactory,
        PiRequestFactory $piRequestFactory,
        PiMerchantInterfaceFactory $piMerchantInterfaceFactory,
        Curl $curl,
        Config $config,
        CacheManager $cacheManager
    ) {
        parent::__construct($context);
        $this->storeManager = $storeManager;
        $this->product = $product;
        $this->formkey = $formkey;
        $this->quote = $quote;
        $this->quoteManagement = $quoteManagement;
        $this->customerFactory = $customerFactory;
        $this->customerRepository = $customerRepository;
        $this->orderService = $orderService;
        $this->piManagementInterfaceFactory = $piManagementInterfaceFactory;
        $this->piRequestFactory = $piRequestFactory;
        $this->piMerchantInterfaceFactory = $piMerchantInterfaceFactory;
        $this->curl = $curl;
        $this->config = $config;
        $this->cacheManager = $cacheManager;
    }

    /**
     * create order programmatically in Magento 2
     *
     * @param array $orderData
     * @return array
     * @throws \Magento\Framework\Exception\LocalizedException
     * @throws \Magento\Framework\Exception\NoSuchEntityException
     */
    public function createOrder($orderData, $customerData, $shouldLoadQuote = false)
    {
        $this->flushCache();
        /** @var Store $store */
        $store = $this->storeManager->getStore();

        //Create quote object
        /** @var Quote $quote */
        $quote = $this->quote->create();

        //set store for which you create quote
        $quote->setStore($store);
        $quote->setCurrency();

        if ($shouldLoadQuote) {
            $customer = $this->customerFactory->create()->setWebsiteId(1)->loadByEmail($customerData['email']);
            $quote->loadByCustomer($customer);
        } else {
            // Set Customer Data on Quote, Do not create customer.
            $quote->setCustomerFirstname($customerData['firstname']);
            $quote->setCustomerLastname($customerData['lastname']);
            $quote->setCustomerEmail($customerData['email']);
            $quote->setCustomerIsGuest($customerData['isGuest']);

            //add items in quote
            foreach ($orderData['items'] as $item) {
                $product = $this->product->load($item['product_id']);
                $product->setPrice($item['price']);
                $quote->addProduct(
                    $product,
                    intval($item['qty'])
                );
            }
        }

        //Set Address to quote
        $quote->getBillingAddress()->addData($orderData['shipping_address']);
        $quote->getShippingAddress()->addData($orderData['shipping_address']);

        // Collect Rates and Set Shipping & Payment Method

        $shippingAddress = $quote->getShippingAddress();
        // set shipping method
        $shippingAddress->setCollectShippingRates(true)
            ->collectShippingRates()
            ->setShippingMethod('flatrate_flatrate');

        // set payment method
        $quote->setPaymentMethod('sagepaysuitepi');

        // used to manage inventory, if set true then it update inventory after successful order.
        $quote->setInventoryProcessed(false);

        // Save quote
        $quote->save();

        // Set Sales Order Payment
        $quote->getPayment()->importData(['method' => 'sagepaysuitepi']);

        // Collect Totals & Save Quote
        $quote->collectTotals()->save();

        $this->flushCache();
        // generate merchantSessionkey
        /** @var PiMerchantInterface $piMerchantInterface */
        $piMerchantInterface = $this->piMerchantInterfaceFactory->create();
        /** @var ResultInterface $merchantSessionKey */
        $merchantSessionKeyResponse = $piMerchantInterface->getSessionKey($quote);
        $merchantSessionKey = $merchantSessionKeyResponse->getResponse();

        // generate card identifier
        $cardIdentifier = $this->getCardIdentifier($merchantSessionKey);
        // set piRequestInterface
        /** @var PiRequest $piRequest */
        $piRequest = $this->piRequestFactory->create();
        $piRequest->setCardIdentifier($cardIdentifier);
        $piRequest->setMerchantSessionKey($merchantSessionKey);
        $piRequest->setCcLastFour('');
        $piRequest->setCcExpMonth('');
        $piRequest->setCcExpYear('');
        $piRequest->setCcType('');
        $piRequest->setSaveToken(false);
        $piRequest->setReusableToken(false);
        $piRequest->setJavascriptEnabled(1);
        $piRequest->setAcceptHeaders("*\/*");
        $piRequest->setLanguage("en-GB");
        $piRequest->setUserAgent("Mozilla\/5.0 (Macintosh; Intel Mac OS X 10_15_7)".
        " AppleWebKit\/537.36 (KHTML, like Gecko) Chrome\/108.0.0.0 Safari\/537.36");
        $piRequest->setJavaEnabled(0);
        $piRequest->setColorDepth(24);
        $piRequest->setScreenWidth(1440);
        $piRequest->setScreenHeight(900);
        $piRequest->setTimezone(180);

        // Create Order From Quote
        /** @var PiManagementInterface $piManagementInterface */
        $piManagementInterface = $this->piManagementInterfaceFactory->create();
        /** @var PiResultInterface $piRequestResult */
        $piRequestResult = $piManagementInterface->savePaymentInformationAndPlaceOrder($quote->getId(), $piRequest);
        $this->flushCache();

        if ($piRequestResult->getOrderId()) {
            $result['order_id']= $piRequestResult->getOrderId();
        } else {
            $result=['error'=>1,'msg'=>'Error while creating order'];
        }

        return $result;
    }

    /**
     * @param $merchantSessionKey
     * @return mixed
     */
    private function getCardIdentifier($merchantSessionKey)
    {
        $payload = [
            "cardDetails" => [
                "cardholderName" => "status201ds",
                "cardNumber" => self::TEST_CC_NUMBER,
                "expiryDate" => self::TEST_CC_EXPIRY,
                "securityCode" => self::TEST_CC_CV2,
            ]
        ];

        // @codingStandardsIgnoreStart
        $this->curl->write(
            HttpRequest::METHOD_POST,
            "https://sandbox.opayo.eu.elavon.com/api/v1/card-identifiers", //http because of proxy
            HttpRequest::VERSION_11,
            ["Content-type: application/json", "Authorization: Bearer $merchantSessionKey", "Cache-Control: no-cache"],
            json_encode($payload)
        );
        // @codingStandardsIgnoreEnd

        $cardIdentifierResponseBody = $this->curl->read();
        $cardIdentifierResponse = HttpResponse::fromString($cardIdentifierResponseBody)->getBody(); // @codingStandardsIgnoreLine
        $cardIdentifierResponseObject = json_decode($cardIdentifierResponse);

        return $cardIdentifierResponseObject->cardIdentifier;
    }

    private function flushCache()
    {
        $this->cacheManager->flush($this->cacheManager->getAvailableTypes());
    }
}
